/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.pdfbox.tools;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;

import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;

import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.font.PDFont;
import org.apache.pdfbox.pdmodel.font.PDType0Font;
import org.apache.pdfbox.pdmodel.font.PDType1Font;
import org.apache.pdfbox.util.Charsets;

/**
 * This will take a text file and output a pdf with that text.
 *
 * @author Ben Litchfield
 */
public class TextToPDF
{
    /**
     * The scaling factor for font units to PDF units
     */
    private static final int FONTSCALE = 1000;
    
    /**
     * The default font
     */
    private static final PDType1Font DEFAULT_FONT = PDType1Font.HELVETICA;

    /**
     * The default font size
     */
    private static final float DEFAULT_FONT_SIZE = 10;
    
    /**
     * The line height as a factor of the font size
     */
    private static final float DEFAULT_LINE_HEIGHT_FACTOR = 1.05f;
    
    /**
     * The default margin
     */
    private static final float DEFAULT_MARGIN = 40;

    private float fontSize = DEFAULT_FONT_SIZE;
    private float lineSpacing = DEFAULT_LINE_HEIGHT_FACTOR;
    private PDRectangle mediaBox = PDRectangle.LETTER;
    private boolean landscape = false;
    private PDFont font = DEFAULT_FONT;
    private float leftMargin = DEFAULT_MARGIN;
    private float rightMargin = DEFAULT_MARGIN;
    private float topMargin = DEFAULT_MARGIN;
    private float bottomMargin = DEFAULT_MARGIN;

    private static final Map<String, PDType1Font> STANDARD_14 = new HashMap<String, PDType1Font>();
    static
    {
        STANDARD_14.put(PDType1Font.TIMES_ROMAN.getBaseFont(), PDType1Font.TIMES_ROMAN);
        STANDARD_14.put(PDType1Font.TIMES_BOLD.getBaseFont(), PDType1Font.TIMES_BOLD);
        STANDARD_14.put(PDType1Font.TIMES_ITALIC.getBaseFont(), PDType1Font.TIMES_ITALIC);
        STANDARD_14.put(PDType1Font.TIMES_BOLD_ITALIC.getBaseFont(), PDType1Font.TIMES_BOLD_ITALIC);
        STANDARD_14.put(PDType1Font.HELVETICA.getBaseFont(), PDType1Font.HELVETICA);
        STANDARD_14.put(PDType1Font.HELVETICA_BOLD.getBaseFont(), PDType1Font.HELVETICA_BOLD);
        STANDARD_14.put(PDType1Font.HELVETICA_OBLIQUE.getBaseFont(), PDType1Font.HELVETICA_OBLIQUE);
        STANDARD_14.put(PDType1Font.HELVETICA_BOLD_OBLIQUE.getBaseFont(), PDType1Font.HELVETICA_BOLD_OBLIQUE);
        STANDARD_14.put(PDType1Font.COURIER.getBaseFont(), PDType1Font.COURIER);
        STANDARD_14.put(PDType1Font.COURIER_BOLD.getBaseFont(), PDType1Font.COURIER_BOLD);
        STANDARD_14.put(PDType1Font.COURIER_OBLIQUE.getBaseFont(), PDType1Font.COURIER_OBLIQUE);
        STANDARD_14.put(PDType1Font.COURIER_BOLD_OBLIQUE.getBaseFont(), PDType1Font.COURIER_BOLD_OBLIQUE);
        STANDARD_14.put(PDType1Font.SYMBOL.getBaseFont(), PDType1Font.SYMBOL);
        STANDARD_14.put(PDType1Font.ZAPF_DINGBATS.getBaseFont(), PDType1Font.ZAPF_DINGBATS);
    }

    /**
     * Create a PDF document with some text.
     *
     * @param text The stream of text data.
     *
     * @return The document with the text in it.
     *
     * @throws IOException If there is an error writing the data.
     */
    public PDDocument createPDFFromText( Reader text ) throws IOException
    {
        PDDocument doc = new PDDocument();
        createPDFFromText(doc, text);
        return doc;
    }

    /**
     * Create a PDF document with some text.
     *
     * @param doc The document.
     * @param text The stream of text data.
     *
     * @throws IOException If there is an error writing the data.
     */
    public void createPDFFromText( PDDocument doc, Reader text ) throws IOException
    {
        float fontHeight = font.getBoundingBox().getHeight() / FONTSCALE;
        PDRectangle actualMediaBox =
                landscape ? new PDRectangle(mediaBox.getHeight(), mediaBox.getWidth()) : mediaBox;

        // calculate line height and increase by a factor.
        float lineHeight = fontHeight * fontSize * lineSpacing;
        BufferedReader data = new BufferedReader(text);
        String nextLine;
        PDPage page = new PDPage(actualMediaBox);
        PDPageContentStream contentStream = null;
        float y = -1;
        float maxStringLength = page.getMediaBox().getWidth() - leftMargin - rightMargin;

        // There is a special case of creating a PDF document from an empty string.
        boolean textIsEmpty = true;

        StringBuilder nextLineToDraw = new StringBuilder();

        while ((nextLine = data.readLine()) != null)
        {
            // The input text is nonEmpty. New pages will be created and added
            // to the PDF document as they are needed, depending on the length of
            // the text.
            textIsEmpty = false;

            String[] lineWords = nextLine.replaceAll("[\\n\\r]+$", "").split(" ", -1);
            int lineIndex = 0;
            while (lineIndex < lineWords.length)
            {
                nextLineToDraw.setLength(0);
                boolean addSpace = false;
                float lengthIfUsingNextWord = 0;
                boolean ff = false;
                do
                {
                    String word1, word2 = "";
                    String word = lineWords[lineIndex];
                    int indexFF = word.indexOf('\f');
                    if (indexFF == -1)
                    {
                        word1 = word;
                    }
                    else
                    {
                        ff = true;
                        word1 = word.substring(0, indexFF);
                        if (indexFF < word.length())
                        {
                            word2 = word.substring(indexFF + 1);
                        }
                    }
                    // word1 is the part before ff, word2 after
                    // both can be empty
                    // word1 can also be empty without ff, if a line has many spaces
                    if (word1.length() > 0 || !ff)
                    {
                        if (addSpace)
                        {
                            nextLineToDraw.append(' ');
                        }
                        else
                        {
                            addSpace = true;
                        }
                        nextLineToDraw.append(word1);
                    }
                    if (!ff || word2.length() == 0)
                    {
                        lineIndex++;
                    }
                    else
                    {
                        lineWords[lineIndex] = word2;
                    }
                    if (ff)
                    {
                        break;
                    }
                    if (lineIndex < lineWords.length)
                    {
                        // need cut off at \f in next word to avoid IllegalArgumentException
                        String nextWord = lineWords[lineIndex];
                        indexFF = nextWord.indexOf('\f');
                        if (indexFF != -1)
                        {
                            nextWord = nextWord.substring(0, indexFF);
                        }

                        String lineWithNextWord = nextLineToDraw + " " + nextWord;
                        lengthIfUsingNextWord
                                = (font.getStringWidth(lineWithNextWord) / FONTSCALE) * fontSize;
                    }
                }
                while (lineIndex < lineWords.length && lengthIfUsingNextWord < maxStringLength);

                if (y - lineHeight < bottomMargin)
                {
                    // We have crossed the end-of-page boundary and need to extend the
                    // document by another page.
                    page = new PDPage(actualMediaBox);
                    doc.addPage(page);
                    if (contentStream != null)
                    {
                        contentStream.endText();
                        contentStream.close();
                    }
                    contentStream = new PDPageContentStream(doc, page);
                    contentStream.setFont(font, fontSize);
                    contentStream.beginText();
                    y = page.getMediaBox().getHeight() - topMargin;
                    y += lineHeight - fontHeight * fontSize; // adjust for lineSpacing != 1
                    contentStream.newLineAtOffset(leftMargin, y);
                }

                if (contentStream == null)
                {
                    throw new IOException("Error:Expected non-null content stream.");
                }
                contentStream.newLineAtOffset(0, -lineHeight);
                y -= lineHeight;
                contentStream.showText(nextLineToDraw.toString());
                if (ff)
                {
                    page = new PDPage(actualMediaBox);
                    doc.addPage(page);
                    contentStream.endText();
                    contentStream.close();
                    contentStream = new PDPageContentStream(doc, page);
                    contentStream.setFont(font, fontSize);
                    contentStream.beginText();
                    y = page.getMediaBox().getHeight() - topMargin;
                    y += lineHeight - fontHeight * fontSize; // adjust for lineSpacing != 1
                    contentStream.newLineAtOffset(leftMargin, y);
                }
            }
        }

        // If the input text was the empty string, then the above while loop will have short-circuited
        // and we will not have added any PDPages to the document.
        // So in order to make the resultant PDF document readable by Adobe Reader etc, we'll add an empty page.
        if (textIsEmpty)
        {
            doc.addPage(page);
        }

        if (contentStream != null)
        {
            contentStream.endText();
            contentStream.close();
        }
    }

    /**
     * This will create a PDF document with some text in it.
     * <br>
     * see usage() for commandline
     *
     * @param args Command line arguments.
     *
     * @throws IOException If there is an error with the PDF.
     */
    public static void main(String[] args) throws IOException
    {
        Charset charset = Charset.defaultCharset();

        // suppress the Dock icon on OS X
        System.setProperty("apple.awt.UIElement", "true");

        TextToPDF app = new TextToPDF();
                
        PDDocument doc = new PDDocument();
        try
        {
            if( args.length < 2 )
            {
                app.usage();
            }
            else
            {
                for( int i=0; i<args.length-2; i++ )
                {
                    if( args[i].equals( "-standardFont" ))
                    {
                        i++;
                        app.setFont( getStandardFont( args[i] ));
                    }
                    else if( args[i].equals( "-ttf" ))
                    {
                        i++;
                        PDFont font = PDType0Font.load( doc, new File( args[i]) );
                        app.setFont( font );
                    }
                    else if( args[i].equals( "-charset" ))
                    {
                        i++;
                        charset = Charset.forName(args[i]);
                    }
                    else if( args[i].equals( "-fontSize" ))
                    {
                        i++;
                        app.setFontSize( Float.parseFloat( args[i] ) );
                    }
                    else if (args[i].equals("-margins"))
                    {
                        i++;
                        app.setLeftMargin(Float.parseFloat(args[i]));
                        i++;
                        app.setRightMargin(Float.parseFloat(args[i]));
                        i++;
                        app.setTopMargin(Float.parseFloat(args[i]));
                        i++;
                        app.setBottomMargin(Float.parseFloat(args[i]));
                    }
                    else if( args[i].equals( "-lineSpacing" ))
                    {
                        i++;
                        app.setLineSpacing( Float.parseFloat( args[i] ) );
                    }
                    else if( args[i].equals( "-pageSize" ))
                    {
                        i++;
                        PDRectangle rectangle = createRectangle(args[i]);
                        if (rectangle == null)
                        {
                            throw new IOException("Unknown argument: " + args[i]);
                        }
                        app.setMediaBox(rectangle);
                    }
                    else if( args[i].equals( "-landscape" ))
                    {
                        app.setLandscape(true);
                    }                    
                    else
                    {
                        throw new IOException( "Unknown argument: " + args[i] );
                    }
                }

                boolean hasUtf8BOM = false;
                if (charset.equals(Charsets.UTF_8))
                {
                    // check for utf8 BOM
                    // FileInputStream doesn't support mark/reset
                    InputStream is = new FileInputStream(args[args.length - 1]);
                    if (is.read() == 0xEF && is.read() == 0xBB && is.read() == 0xBF)
                    {
                        hasUtf8BOM = true;
                    }
                    is.close();
                }
                InputStream is = new FileInputStream(args[args.length - 1]);
                if (hasUtf8BOM)
                {
                    long skipped = is.skip(3);
                    if (skipped != 3)
                    {
                        throw new IOException("Could not skip 3 bytes, size changed?!");
                    }
                }
                Reader reader = new InputStreamReader(is, charset);
                app.createPDFFromText(doc, reader);
                reader.close();
                is.close();
                doc.save(args[args.length - 2]);
            }
        }
        finally
        {
            doc.close();
        }
    }

    private static PDRectangle createRectangle( String paperSize )
    {
        if ("letter".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.LETTER;
        }
        else if ("legal".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.LEGAL;
        }
        else if ("A0".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.A0;
        }
        else if ("A1".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.A1;
        }
        else if ("A2".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.A2;
        }
        else if ("A3".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.A3;
        }
    	else if ("A4".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.A4;
        }
        else if ("A5".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.A5;
        }
        else if ("A6".equalsIgnoreCase(paperSize))
        {
            return PDRectangle.A6;
        }
        else
        {
            return null;
        }
    }

    /**
     * This will print out a message telling how to use this example.
     */
    private void usage()
    {
        String[] std14 = getStandard14Names();
        
        StringBuilder message = new StringBuilder();       
        message.append("Usage: jar -jar pdfbox-app-x.y.z.jar TextToPDF [options] <outputfile> <textfile>\n");
        message.append("\nOptions:\n");
        message.append("  -standardFont <name> : ").append(DEFAULT_FONT.getBaseFont()).append(" (default)\n");

        for (String std14String : std14)
        {
            message.append("                         ").append(std14String).append("\n");
        }
        message.append("  -ttf <ttf file>      : The TTF font to use.\n");
        message.append("  -charset <charset>   : default: ").append(Charset.defaultCharset()).append("\n");
        message.append("  -fontSize <fontSize> : default: ").append(DEFAULT_FONT_SIZE).append("\n");
        message.append("  -margins <left> <right> <top> <bottom> : default: ").
                append(DEFAULT_MARGIN).append(" ").append(DEFAULT_MARGIN).append(" ").
                append(DEFAULT_MARGIN).append(" ").append(DEFAULT_MARGIN).append("\n");
        message.append("  -lineSpacing <factor>: default: ").append(DEFAULT_LINE_HEIGHT_FACTOR).append("\n");
        message.append("  -pageSize <pageSize> : Letter (default)\n");
        message.append("                         Legal\n");
        message.append("                         A0\n");
        message.append("                         A1\n");
        message.append("                         A2\n");
        message.append("                         A3\n");
        message.append("                         A4\n");
        message.append("                         A5\n");
        message.append("                         A6\n");
        message.append("  -landscape           : sets orientation to landscape" );

        System.err.println(message.toString());
        System.exit(1);
    }


    /**
     * A convenience method to get one of the standard 14 font from name.
     *
     * @param name The name of the font to get.
     *
     * @return The font that matches the name or null if it does not exist.
     */
    private static PDType1Font getStandardFont(String name)
    {
        return STANDARD_14.get(name);
    }

    /**
     * This will get the names of the standard 14 fonts.
     *
     * @return An array of the names of the standard 14 fonts.
     */
    private static String[] getStandard14Names()
    {
        return STANDARD_14.keySet().toArray(new String[14]);
    }


    /**
     * @return Returns the font.
     */
    public PDFont getFont()
    {
        return font;
    }
    /**
     * @param aFont The font to set.
     */
    public void setFont(PDFont aFont)
    {
        this.font = aFont;
    }

    /**
     * @return Returns the fontSize, truncated to integer.
     */
    public int getFontSize()
    {
        return (int) fontSize;
    }

    /**
     * @param aFontSize The fontSize to set.
     * 
     * @deprecated use {@link #setFontSize(float)}
     */
    @Deprecated
    public void setFontSize(int aFontSize)
    {
        this.fontSize = aFontSize;
    }

    /**
     * @param aFontSize The fontSize to set.
     */
    public void setFontSize(float aFontSize)
    {
        this.fontSize = aFontSize;
    }

    /**
     * @return Returns the lineSpacing.
     */
    public float getLineSpacing()
    {
        return lineSpacing;
    }

    /**
     * @param lineSpacing The lineSpacing to set.
     */
    public void setLineSpacing(float lineSpacing)
    {
        if (lineSpacing <= 0)
        {
            throw new IllegalArgumentException("line spacing must be positive: " + lineSpacing);
        }
        this.lineSpacing = lineSpacing;
    }

    /**
     * @return Returns the left margin.
     */
    public float getLeftMargin()
    {
        return leftMargin;
    }

    /**
     * @param leftMargin The left margin to be set.
     */
    public void setLeftMargin(float leftMargin)
    {
        this.leftMargin = leftMargin;
    }

    /**
     * @return Returns the right margin.
     */
    public float getRightMargin()
    {
        return rightMargin;
    }

    /**
     * @param rightMargin The right margin to be set.
     */
    public void setRightMargin(float rightMargin)
    {
        this.rightMargin = rightMargin;
    }

    /**
     * @return Returns the top margin.
     */
    public float getTopMargin()
    {
        return topMargin;
    }

    /**
     * @param topMargin The top margin to be set.
     */
    public void setTopMargin(float topMargin)
    {
        this.topMargin = topMargin;
    }

    /**
     * @return Returns the bottom margin.
     */
    public float getBottomMargin()
    {
        return bottomMargin;
    }

    /**
     * @param bottomMargin The bottom margin to be set.
     */
    public void setBottomMargin(float bottomMargin)
    {
        this.bottomMargin = bottomMargin;
    }

    /**
     * Sets page size of produced PDF.
     *
     * @return returns the page size (media box)
     */
    public PDRectangle getMediaBox()
    {
        return mediaBox;
    }

    /**
     * Sets page size of produced PDF.
     *
     * @param mediaBox
     */
    public void setMediaBox(PDRectangle mediaBox)
    {
        this.mediaBox = mediaBox;
    }

    /**
     * Tells the paper orientation.
     *
     * @return true for landscape orientation
     */
    public boolean isLandscape()
    {
        return landscape;
    }

    /**
     * Sets paper orientation.
     *
     * @param landscape true for landscape orientation
     */
    public void setLandscape(boolean landscape)
    {
        this.landscape = landscape;
    }
}
