/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.common.blobstore.fs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import org.opensearch.common.UUIDs;
import org.opensearch.common.blobstore.BlobContainer;
import org.opensearch.common.blobstore.BlobMetadata;
import org.opensearch.common.blobstore.BlobPath;
import org.opensearch.common.blobstore.DeleteResult;
import org.opensearch.common.blobstore.fs.FsBlobStore;
import org.opensearch.common.blobstore.support.AbstractBlobContainer;
import org.opensearch.common.blobstore.support.PlainBlobMetadata;
import org.opensearch.core.internal.io.IOUtils;
import org.opensearch.core.internal.io.Streams;

public class FsBlobContainer
extends AbstractBlobContainer {
    private static final String TEMP_FILE_PREFIX = "pending-";
    protected final FsBlobStore blobStore;
    protected final Path path;

    public FsBlobContainer(FsBlobStore blobStore, BlobPath blobPath, Path path) {
        super(blobPath);
        this.blobStore = blobStore;
        this.path = path;
    }

    @Override
    public Map<String, BlobMetadata> listBlobs() throws IOException {
        return this.listBlobsByPrefix(null);
    }

    @Override
    public Map<String, BlobContainer> children() throws IOException {
        HashMap<String, FsBlobContainer> builder = new HashMap<String, FsBlobContainer>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.path);){
            for (Path file : stream) {
                if (!Files.isDirectory(file, new LinkOption[0])) continue;
                String name = file.getFileName().toString();
                builder.put(name, new FsBlobContainer(this.blobStore, this.path().add(name), file));
            }
        }
        return Collections.unmodifiableMap(builder);
    }

    @Override
    public Map<String, BlobMetadata> listBlobsByPrefix(String blobNamePrefix) throws IOException {
        HashMap<String, PlainBlobMetadata> builder = new HashMap<String, PlainBlobMetadata>();
        blobNamePrefix = blobNamePrefix == null ? "" : blobNamePrefix;
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.path, blobNamePrefix + "*");){
            for (Path file : stream) {
                BasicFileAttributes attrs;
                try {
                    attrs = Files.readAttributes(file, BasicFileAttributes.class, new LinkOption[0]);
                }
                catch (FileNotFoundException | NoSuchFileException e) {
                    continue;
                }
                if (!attrs.isRegularFile()) continue;
                builder.put(file.getFileName().toString(), new PlainBlobMetadata(file.getFileName().toString(), attrs.size()));
            }
        }
        return Collections.unmodifiableMap(builder);
    }

    @Override
    public DeleteResult delete() throws IOException {
        final AtomicLong filesDeleted = new AtomicLong(0L);
        final AtomicLong bytesDeleted = new AtomicLong(0L);
        Files.walkFileTree(this.path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException impossible) throws IOException {
                assert (impossible == null);
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                filesDeleted.incrementAndGet();
                bytesDeleted.addAndGet(attrs.size());
                return FileVisitResult.CONTINUE;
            }
        });
        return new DeleteResult(filesDeleted.get(), bytesDeleted.get());
    }

    @Override
    public void deleteBlobsIgnoringIfNotExists(List<String> blobNames) throws IOException {
        IOUtils.rm((Path[])((Path[])blobNames.stream().map(this.path::resolve).toArray(Path[]::new)));
    }

    @Override
    public boolean blobExists(String blobName) {
        return Files.exists(this.path.resolve(blobName), new LinkOption[0]);
    }

    @Override
    public InputStream readBlob(String name) throws IOException {
        Path resolvedPath = this.path.resolve(name);
        try {
            return Files.newInputStream(resolvedPath, new OpenOption[0]);
        }
        catch (FileNotFoundException fnfe) {
            throw new NoSuchFileException("[" + name + "] blob not found");
        }
    }

    @Override
    public InputStream readBlob(String blobName, long position, long length) throws IOException {
        SeekableByteChannel channel = Files.newByteChannel(this.path.resolve(blobName), new OpenOption[0]);
        if (position > 0L) {
            channel.position(position);
        }
        assert (channel.position() == position);
        return org.opensearch.common.io.Streams.limitStream(Channels.newInputStream(channel), length);
    }

    @Override
    public long readBlobPreferredLength() {
        return Long.MAX_VALUE;
    }

    @Override
    public void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        Path file = this.path.resolve(blobName);
        try {
            this.writeToPath(inputStream, file, blobSize);
        }
        catch (FileAlreadyExistsException faee) {
            if (failIfAlreadyExists) {
                throw faee;
            }
            this.deleteBlobsIgnoringIfNotExists(Collections.singletonList(blobName));
            this.writeToPath(inputStream, file, blobSize);
        }
        IOUtils.fsync((Path)this.path, (boolean)true);
    }

    @Override
    public void writeBlobAtomic(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        String tempBlob = FsBlobContainer.tempBlobName(blobName);
        Path tempBlobPath = this.path.resolve(tempBlob);
        try {
            this.writeToPath(inputStream, tempBlobPath, blobSize);
            this.moveBlobAtomic(tempBlob, blobName, failIfAlreadyExists);
        }
        catch (IOException ex) {
            try {
                this.deleteBlobsIgnoringIfNotExists(Collections.singletonList(tempBlob));
            }
            catch (IOException e) {
                ex.addSuppressed(e);
            }
            throw ex;
        }
        finally {
            IOUtils.fsync((Path)this.path, (boolean)true);
        }
    }

    private void writeToPath(InputStream inputStream, Path tempBlobPath, long blobSize) throws IOException {
        try (OutputStream outputStream = Files.newOutputStream(tempBlobPath, StandardOpenOption.CREATE_NEW);){
            int bufferSize = this.blobStore.bufferSizeInBytes();
            Streams.copy((InputStream)inputStream, (OutputStream)outputStream, (byte[])new byte[blobSize < (long)bufferSize ? Math.toIntExact(blobSize) : bufferSize]);
        }
        IOUtils.fsync((Path)tempBlobPath, (boolean)false);
    }

    public void moveBlobAtomic(String sourceBlobName, String targetBlobName, boolean failIfAlreadyExists) throws IOException {
        Path sourceBlobPath = this.path.resolve(sourceBlobName);
        Path targetBlobPath = this.path.resolve(targetBlobName);
        if (Files.exists(targetBlobPath, new LinkOption[0])) {
            if (failIfAlreadyExists) {
                throw new FileAlreadyExistsException("blob [" + targetBlobPath + "] already exists, cannot overwrite");
            }
            this.deleteBlobsIgnoringIfNotExists(Collections.singletonList(targetBlobName));
        }
        Files.move(sourceBlobPath, targetBlobPath, StandardCopyOption.ATOMIC_MOVE);
    }

    public static String tempBlobName(String blobName) {
        return TEMP_FILE_PREFIX + blobName + "-" + UUIDs.randomBase64UUID();
    }

    public static boolean isTempBlobName(String blobName) {
        return blobName.startsWith(TEMP_FILE_PREFIX);
    }
}

