/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.bean.openapi3;

import static org.apache.juneau.TestUtils.*;
import static org.apache.juneau.bean.openapi3.OpenApiBuilder.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.junit.bct.BctAssertions.*;
import static org.junit.jupiter.api.Assertions.*;

import java.net.*;

import org.apache.juneau.*;
import org.junit.jupiter.api.*;

/**
 * Testcase for {@link OpenApi}.
 */
class OpenApi_Test extends TestBase {

	@Nested class A_basicTests extends TestBase {

		private static final BeanTester<OpenApi> TESTER =
			testBean(
				bean()
					.setComponents(components().setSchemas(map("a1", schemaInfo().setType("a2"))))
					.setExternalDocs(externalDocumentation().setUrl(URI.create("b")))
					.setInfo(info().setTitle("c1").setVersion("c2"))
					.setOpenapi("3.0.0")
					.setPaths(map("d1", pathItem().setGet(operation().setSummary("d2"))))
					.setSecurity(l(securityRequirement().setRequirements(map("e1",l("e2")))))
					.setServers(l(server().setUrl(URI.create("f"))))
					.setTags(l(tag().setName("g")))
			)
			.props("components{schemas{a1{type}}},externalDocs{url},info{title,version},openapi,paths{d1{get{summary}}},security{0{requirements{e1}}},servers{0{url}},tags{0{name}}")
			.vals("{{{a2}}},{b},{c1,c2},3.0.0,{{{d2}}},{{{[e2]}}},{{f}},{{g}}")
			.json("{components:{schemas:{a1:{type:'a2'}}},externalDocs:{url:'b'},info:{title:'c1',version:'c2'},openapi:'3.0.0',paths:{d1:{get:{summary:'d2'}}},security:[{requirements:{e1:['e2']}}],servers:[{url:'f'}],tags:[{name:'g'}]}")
			.string("{'components':{'schemas':{'a1':{'type':'a2'}}},'externalDocs':{'url':'b'},'info':{'title':'c1','version':'c2'},'openapi':'3.0.0','paths':{'d1':{'get':{'summary':'d2'}}},'security':[{'requirements':{'e1':['e2']}}],'servers':[{'url':'f'}],'tags':[{'name':'g'}]}".replace('\'','"'))
		;

		@Test void a01_gettersAndSetters() {
			TESTER.assertGettersAndSetters();
		}

		@Test void a02_copy() {
			TESTER.assertCopy();
		}

		@Test void a03_toJson() {
			TESTER.assertToJson();
		}

		@Test void a04_fromJson() {
			TESTER.assertFromJson();
		}

		@Test void a05_roundTrip() {
			TESTER.assertRoundTrip();
		}

		@Test void a06_toString() {
			TESTER.assertToString();
		}

		@Test void a07_keySet() {
			assertList(TESTER.bean().keySet(), "components", "externalDocs", "info", "openapi", "paths", "security", "servers", "tags");
		}

		@Test void a08_nullParameters() {
			var x = bean();
			assertThrows(IllegalArgumentException.class, () -> x.get(null, String.class));
			assertThrows(IllegalArgumentException.class, () -> x.set(null, "value"));
		}

		@Test void a09_addMethods() {
			assertBean(
				bean()
					.addPath("/test", pathItem().setGet(operation().setSummary("a"))),
				"paths{/test{get{summary}}}",
				"{{{a}}}"
			);

			// Test addTags
			var x1 = bean()
				.addTags(tag("b1"), tag("b2"))
				.addTags(l(tag("b3")));
			assertBean(x1, "tags{#{name}}", "{[{b1},{b2},{b3}]}");

			// Test addServers
			var x2 = bean()
				.addServers(server().setUrl(URI.create("http://c1.com")), server().setUrl(URI.create("http://c2.com")))
				.addServers(l(server().setUrl(URI.create("http://c3.com"))));
			assertBean(x2, "servers{#{url}}", "{[{http://c1.com},{http://c2.com},{http://c3.com}]}");

			// Test addSecurity
			var x3 = bean()
				.addSecurity(securityRequirement().setRequirements(map("d1", l("d2"))))
				.addSecurity(l(securityRequirement().setRequirements(map("d3", l("d4")))));
			assertBean(x3, "security{#{requirements}}", "{[{{d1=[d2]}},{{d3=[d4]}}]}");
		}

		@Test void a09b_setTagsVarargs() {
			// Test setTags(Tag...) varargs method to cover that code path
			var x = bean().setTags(tag("t1"), tag("t2"), tag("t3"));
			assertBean(x, "tags{#{name}}", "{[{t1},{t2},{t3}]}");
		}

		@Test void a10_asMap() {
			assertBean(
				bean()
					.setInfo(info().setTitle("a1").setVersion("a2"))
					.set("x1", "x1a")
					.asMap(),
				"info{title,version},openapi,x1",
				"{a1,a2},3.0.0,x1a"
			);
		}

		@Test void a11_extraKeys() {
			var x = bean().set("x1", "x1a").set("x2", "x2a");
			assertList(x.extraKeys(), "x1", "x2");
			assertEmpty(bean().extraKeys());
		}

		@Test void a12_strictMode() {
			assertThrows(RuntimeException.class, () -> bean().strict().set("foo", "bar"));
			assertDoesNotThrow(() -> bean().set("foo", "bar"));

			assertFalse(bean().isStrict());
			assertTrue(bean().strict().isStrict());
			assertFalse(bean().strict(false).isStrict());
		}

		@Test void a13_collectionSetters() {
			var x = bean()
				.setSecurity(l(securityRequirement().setRequirements(map("a1", l("a2"))), securityRequirement().setRequirements(map("b1", l("b2")))))
				.setServers(l(server().setUrl(URI.create("http://example1.com")), server().setUrl(URI.create("http://example2.com"))))
				.setTags(l(tag().setName("c1"), tag().setName("c2")));

			assertBean(x,
				"security{#{requirements}},servers{#{url}},tags{#{name}}",
				"{[{{a1=[a2]}},{{b1=[b2]}}]},{[{http://example1.com},{http://example2.com}]},{[{c1},{c2}]}"
			);
		}

		@Test void a14_addPath() {
			// Test addPath method
			var x = bean()
				.addPath("/test", pathItem().setGet(operation().setSummary("Test operation")))
				.addPath("/test2", pathItem().setPost(operation().setSummary("Test POST operation")));

			assertBean(x,
				"paths{/test{get{summary}},/test2{post{summary}}}",
				"{{{Test operation}},{{Test POST operation}}}"
			);

			// Test addPath with null path (covers the null check branch)
			assertThrows(IllegalArgumentException.class, () -> x.addPath(null, pathItem()));
			assertThrows(IllegalArgumentException.class, () -> x.addPath("/test", null));
		}
	}

	@Nested class B_emptyTests extends TestBase {

		private static final BeanTester<OpenApi> TESTER =
			testBean(bean())
			.props("openapi,info,externalDocs,servers,tags,paths,components,security")
			.vals("3.0.0,<null>,<null>,<null>,<null>,<null>,<null>,<null>")
			.json("{openapi:'3.0.0'}")
			.string("{\"openapi\":\"3.0.0\"}")
		;

		@Test void b01_gettersAndSetters() {
			TESTER.assertGettersAndSetters();
		}

		@Test void b02_copy() {
			TESTER.assertCopy();
		}

		@Test void b03_toJson() {
			TESTER.assertToJson();
		}

		@Test void b04_fromJson() {
			TESTER.assertFromJson();
		}

		@Test void b05_roundTrip() {
			TESTER.assertRoundTrip();
		}

		@Test void b06_toString() {
			TESTER.assertToString();
		}

		@Test void b07_keySet() {
			assertList(TESTER.bean().keySet(), "openapi");
		}

		@Test void b08_keySet_infoNull() {
			// Explicitly test keySet when info is null to cover that branch
			var x = bean().setOpenapi("3.0.0"); // info is null
			assertList(x.keySet(), "openapi");
			assertFalse(x.keySet().contains("info"));
		}
	}

	@Nested class C_extraProperties extends TestBase {
		private static final BeanTester<OpenApi> TESTER =
			testBean(
				bean()
					.set("components", components().setSchemas(map("a1", schemaInfo("a2"))))
					.set("externalDocs", externalDocumentation().setUrl(URI.create("b")))
					.set("info", info().setTitle("c").setVersion("d"))
					.set("openapi", "e")
					.set("paths", m("f1", pathItem().setGet(operation().setSummary("f2"))))
					.set("security", l(securityRequirement().setRequirements(map("g1",l("g2")))))
					.set("servers", l(server().setUrl(URI.create("h"))))
					.set("tags", l(tag("i")))
					.set("x1", "x1a")
					.set("x2", null)
			)
			.props("components{schemas{a1{type}}},externalDocs{url},info{title,version},openapi,paths{f1{get{summary}}},security{#{requirements{g1{#{toString}}}}},servers{#{url}},tags{#{name}},x1,x2")
			.vals("{{{a2}}},{b},{c,d},e,{{{f2}}},{[{{{[{g2}]}}}]},{[{h}]},{[{i}]},x1a,<null>")
			.json("{components:{schemas:{a1:{type:'a2'}}},externalDocs:{url:'b'},info:{title:'c',version:'d'},openapi:'e',paths:{f1:{get:{summary:'f2'}}},security:[{requirements:{g1:['g2']}}],servers:[{url:'h'}],tags:[{name:'i'}],x1:'x1a'}")
			.string("{'components':{'schemas':{'a1':{'type':'a2'}}},'externalDocs':{'url':'b'},'info':{'title':'c','version':'d'},'openapi':'e','paths':{'f1':{'get':{'summary':'f2'}}},'security':[{'requirements':{'g1':['g2']}}],'servers':[{'url':'h'}],'tags':[{'name':'i'}],'x1':'x1a'}".replace('\'', '"'))
		;

		@Test void c01_gettersAndSetters() {
			TESTER.assertGettersAndSetters();
		}

		@Test void c02_copy() {
			TESTER.assertCopy();
		}

		@Test void c03_toJson() {
			TESTER.assertToJson();
		}

		@Test void c04_fromJson() {
			TESTER.assertFromJson();
		}

		@Test void c05_roundTrip() {
			TESTER.assertRoundTrip();
		}

		@Test void c06_toString() {
			TESTER.assertToString();
		}

		@Test void c07_keySet() {
			assertList(TESTER.bean().keySet(), "components", "externalDocs", "info", "openapi", "paths", "security", "servers", "tags", "x1", "x2");
		}

		@Test void c08_get() {
			assertMapped(
				TESTER.bean(), (obj,prop) -> obj.get(prop, Object.class),
				"components{schemas{a1{type}}},externalDocs{url},info{title,version},openapi,paths{f1{get{summary}}},security{#{requirements{g1{#{toString}}}}},servers{#{url}},tags{#{name}},x1,x2",
				"{{{a2}}},{b},{c,d},e,{{{f2}}},{[{{{[{g2}]}}}]},{[{h}]},{[{i}]},x1a,<null>"
			);
		}

		@Test void c09_getTypes() {
			assertMapped(
				TESTER.bean(), (obj,prop) -> cns(obj.get(prop, Object.class)),
				"components,externalDocs,info,openapi,paths,security,servers,tags,x1,x2",
				"Components,ExternalDocumentation,Info,String,FilteredMap,ArrayList,ArrayList,ArrayList,String,<null>"
			);
		}

		@Test void c10_nullPropertyValue() {
			assertThrows(IllegalArgumentException.class, ()->bean().get(null));
			assertThrows(IllegalArgumentException.class, ()->bean().get(null, String.class));
			assertThrows(IllegalArgumentException.class, ()->bean().set(null, "a"));
		}
	}

	@Nested class D_utilityMethods extends TestBase {

		@Test void d01_findRef() {
			var x = openApi()
				.setComponents(components().setSchemas(map("a1", schemaInfo().setType("a2"))));

			assertBean(
				x.findRef("#/components/schemas/a1", SchemaInfo.class),
				"type",
				"a2"
			);

			assertNull(x.findRef("#/components/schemas/notfound", SchemaInfo.class));

			assertThrows(IllegalArgumentException.class, () -> x.findRef(null, SchemaInfo.class));
			assertThrows(IllegalArgumentException.class, () -> x.findRef("a", null));
			assertThrows(IllegalArgumentException.class, () -> x.findRef("", SchemaInfo.class));
			assertThrowsWithMessage(RuntimeException.class, "Unsupported reference:  'invalid'", () -> x.findRef("invalid", SchemaInfo.class));
		}
	}

	//---------------------------------------------------------------------------------------------
	// Helper methods
	//---------------------------------------------------------------------------------------------

	private static OpenApi bean() {
		return openApi();
	}
}